/******************************************************/
/* Examples Program For "ET-BASE dsPIC30F2010"  Board */
/* Hardware    : ET-BASE dsPIC30F2010                 */
/* Target MCU  : dsPIC30F2010                         */
/*       	   : X-TAL : 7.3728 MHz                   */
/*             : Run 117.9648MHz                      */
/*             : Selec OSC Mode = XT w/PLL 16x        */
/* Compiler    : MPLAB v8.40 + C30 v3.20B      	      */
/* Last Update : 12/October/2009                      */
/******************************************************/
/* Function    : Example Use MCPWM Control Motor      */
/*             : Read ADC0 --> Adjust Duty PWM1       */
/******************************************************/
/* RB0 = ADC0(Adjust Speed DC-Motor)                  */
/* RE0 = PWM1L                                        */
/* RE1 = PWM1H                                        */ 
/* RE8 = FLTA(Error Stop DC-Motor Control)            */
/******************************************************/
/* Test By Used Module  "ET-MINI I/O : DC-MOTOR"      */
/******************************************************/
/* Connect RB0)(ADC0) ---> VOL1(ADC0 Input)           */
/* Connect RE8(FLTA)  ---> SW1(Error Motor Stop)      */
/* Connect RE1(PWM1H) ---> EN(MINI I/O)               */
/* Connect RD0(DIR1)  ---> IN1(MINI I/O)              */
/* Connect RD1(DIR2)  ---> IN2(MINI I/O)              */
/******************************************************/

#include <p30f2010.h>                                		// dsPIC30F2010 MPU Register
#include "pwm.h"											// Used MCPWM Library Function
#include "adc10.h"											// Used 10 Bit ADC Library Function

/* Setup Configuration For ET-BASE dsPIC30F2010 */
_FOSC(CSW_FSCM_OFF & XT_PLL16);								// Disable Clock Switching,Enable Fail-Salf Clock
                                                            // Clock Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);												// Disable Watchdog 
_FBORPOR(PBOR_OFF & PWRT_64 & MCLR_EN);					    // Disable Brown-Out ,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);										// Code Protect OFF
/* End   Configuration For ET-BASE dsPIC30F2010 */

/* pototype  section */
void init_mcpwm(void);										// Initial MCPWM Function
void init_adc(void);										// Initial RB0 = ADC0 (10 Bit)
void delay(unsigned long int);								// Delay Time Function

/* Strat Main Program Here */
int main(void)
{  
  unsigned int Duty;										// Duty Cycle Buffer

  init_mcpwm();												// Initial MCPWM = 57.6 KHz
  init_adc();												// Initial RB0 = ADC0
  
  // Config RD0,RD1 For Control Direction DC-Motor
  TRISDbits.TRISD0 = 0;										// Config RD0 = Output
  TRISDbits.TRISD1 = 0;										// Config RD1 = Output

  // Control Direction DC-Motor
  // 0 : 0 = Stop
  // 0 : 1 = Left to Right
  // 1 : 0 = Right to Left
  // 1 : 1 = Stop
  LATDbits.LATD0 = 1; 										// ON Direction Motor
  LATDbits.LATD1 = 0; 	

  // Auto Start DC-Motor = MAX Speed
  SetDCMCPWM(1,1024,0);										// PDC1 Register,Duty Cycle,PWMCON2.UDIS=0(Enable Updat PWM1)
  delay(10000);												// Delay For Start Motor

  // Strat Read ADC0 --> Adjust DCPWMC Duty Cycle //
  while(1)													// Loop Continue
  {    
    ADCON1bits.SAMP = 1;				    				// Start Sampling ADC Now
    while(!ADCON1bits.SAMP);								// Wait ADC Sampling Complete
    ConvertADC10();											// Convert Result
 	while(BusyADC10());										// Wait Conver Result Complete    

    Duty = ReadADC10(0);									// Read Result ADC0 to Duty           
    SetDCMCPWM(1,Duty,0);									// PDC1 Register,Duty Cycle,PWMCON2.UDIS=0(Enable Updat PWM1)  
    delay(1000);											// Delay Step Before Next Adjust
  }  
}	  

/********************************/
/* Initial PWM for dsPIC30F2010 */
/*  -> PWM Frequency = 57.6 KHz */
/********************************/
void init_mcpwm()
{		  
  CloseMCPWM();												// Disable MCPWM Before New Config

  // Config MCPWM Interrupt Control
  ConfigIntMCPWM(PWM_INT_DIS &								// Disable PWM Interrupt
    		     PWM_INT_PR6 &								// PWM Interrupt Priority = 6
                 PWM_FLTA_DIS_INT &							// Disable Fault-A Interrupt
				 PWM_FLTA_INT_PR7);							// Fault-A Interrupt Priority = 7

  SetMCPWMFaultA(PWM_OVA1H_INACTIVE & 						// Enable Fault-A Control PWM1H = OFF
                 PWM_FLTA_MODE_LATCH &						// Fault-A Mode = Latch
                 PWM_FLTA1_EN);			    				// Enable Fault-A CH1
				      
  //***************************************************
  // ET-BASE dsPIC30F2010 Hardware Board
  // XTAL = 7.3728MHz
  // Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  // Fcy  = Fosc / 4 
  //      = 117.9648 / 4 = 29.4912 MHz
  // Tcy  = 1 / 29.4912 MHz
  //      = 33.90842 nS
  //***************************************************
  // PWM Clock   = Prescale = 1 
  // 1 Cycle PWM = 1 / 29.4912 MHz 
  // Desire MAX Duty Cycle = 1024 
  // PWM Period = 1024/2
  //            = 512
  // Frequency  = 29.4912 MHz/ 512
  //            = 57.6 KHz
  // Duty Cycle = 1...1024
  //****************************************************

  OpenMCPWM(512,											// PTPER = Period = 57.6 KHz
            0,												// SEVTCMP = Special Time(Not Used) 

			// PTCON
            PWM_EN &										// Enable PWM Function
			PWM_IDLE_STOP &									// Disable PWM in IDLE Mode
            PWM_OP_SCALE1 &	    							// PWM Post Scale = 1
			PWM_IPCLK_SCALE1 &								// PWM Input Clock Prescale = 1
            PWM_MOD_FREE ,									// PWM = Free Running
            
    		// PWMCON1
            PWM_MOD1_IND &									// PWM1 = Free Mode
            PWM_PEN1H,     									// PWM1 High
			
			// PWMCON2	            
			PWM_SEVOPS1 &									// Special Even Post Scaler = 1:1
			PWM_OSYNC_PWM &									// Overide Sync. With PWM Clock
            PWM_UEN);										// Enable PWM Update		  
}

/********************************/
/* Initial ADC for dsPIC30F2010 */
/* Used RB0 = 10 Bit ADC (ADC0) */
/********************************/
void init_adc()
{
  ADCON1bits.ADON = 0;										// Turn-OFF ADC Before Change Config

  // Initial ADC Channel	
  SetChanADC10(ADC_CH0_POS_SAMPLEA_AN0 &					// ADC0 = Input(+)			   
			   ADC_CH0_NEG_SAMPLEA_NVREF);					// GND  = Input(-)             

  // Initial ADC Interrupt
  ConfigIntADC10(ADC_INT_DISABLE);							// Dsiable ADC Interrupt
  
  // Open ADC Function & Turn ON ADC
  OpenADC10(ADC_MODULE_ON &									// Turn-ON ADC Function
       		ADC_IDLE_STOP &									// Stop ADC in IDLE Mode
			ADC_FORMAT_INTG &								// Result Format = Unsigned Integer 
			ADC_CLK_MANUAL &								// ADC Clock = Manual
			ADC_AUTO_SAMPLING_ON &							// Enable ADC Sampling 
            ADC_SAMPLE_SIMULTANEOUS,						// Sample Style = Simultaneous			

            // ADC Config2
			ADC_VREF_AVDD_AVSS &							// VDD=Vref(H),VSS=Vref(L)
			ADC_SCAN_ON &									// Enable Scan        
            ADC_CONVERT_CH0 &								// Used ADC0 to Convert Result
            ADC_SAMPLES_PER_INT_1 &		    				// Number of Sample Between Interrupt
            ADC_ALT_BUF_OFF & 								// Disable Alternate Buffer
			ADC_ALT_INPUT_OFF ,								// Disable Alternate Input	          

			// ADC Config3
			ADC_SAMPLE_TIME_1 &	    						// Samplink Time = Fast
			ADC_CONV_CLK_INTERNAL_RC &						// Used Internal RC Clock Sampling			
			ADC_CONV_CLK_Tcy,								// Conversion Clock = Fast

			// ADC Config Port = Enable RB0 = ADC0
			ENABLE_AN0_ANA , 								// Enable RB0 = ADC0
		            
			// ADC Config Scan = ON ADC0 ,OFF ADC[1..7]
            SKIP_SCAN_AN1 & 								// Disable Scan ADC1
			SKIP_SCAN_AN2 & 								// Disable Scan ADC2
			SKIP_SCAN_AN3 & 								// Disable Scan ADC3
			SKIP_SCAN_AN4 & 								// Disable Scan ADC4
		    SKIP_SCAN_AN5 &									// Disable Scan ADC5
			SKIP_SCAN_AN6 & 								// Disable Scan ADC6
			SKIP_SCAN_AN7);									// Disable Scan ADC7
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}								// Loop Decrease Counter	
}
